"""
Módulo de IA Real — RestaurantAI Pro
Modelos de Machine Learning para:
- Predicción de demanda por plato
- Optimización de precios
- Forecasting de ingresos
- Detección de anomalías en costes
- Scoring de salud financiera

Usa scikit-learn con datos sintéticos generados a partir de los datos reales
del restaurante para entrenar modelos personalizados.
"""

import numpy as np
import math
from typing import Dict, List, Tuple, Optional
from datetime import datetime

try:
    from sklearn.linear_model import LinearRegression, Ridge
    from sklearn.ensemble import RandomForestRegressor, GradientBoostingRegressor
    from sklearn.preprocessing import StandardScaler, PolynomialFeatures
    from sklearn.cluster import KMeans
    from sklearn.metrics import r2_score, mean_absolute_error
    HAS_SKLEARN = True
except ImportError:
    HAS_SKLEARN = False


class RestaurantAIEngine:
    """Motor de IA para análisis predictivo de restaurantes."""

    def __init__(self, datos_restaurante=None):
        self.datos = datos_restaurante
        self.models = {}
        self.scaler = StandardScaler() if HAS_SKLEARN else None
        self.is_trained = False

    def available(self) -> bool:
        return HAS_SKLEARN

    # =========================================================
    # 1. GENERADOR DE DATOS SINTÉTICOS
    # =========================================================

    def _generate_synthetic_data(self, n_months: int = 24) -> Dict:
        """
        Genera datos sintéticos realistas basados en los datos reales del restaurante.
        Simula variación estacional, tendencia y ruido aleatorio.
        """
        if not self.datos or not self.datos.platos:
            return {}

        np.random.seed(42)
        months = np.arange(n_months)

        # Factores estacionales (hostelería España)
        seasonal = np.array([
            0.85, 0.80, 0.90, 0.95, 1.00, 1.10,  # Ene-Jun
            1.15, 1.20, 1.05, 0.95, 0.90, 1.10,  # Jul-Dic
        ])
        seasonal_full = np.tile(seasonal, n_months // 12 + 1)[:n_months]

        # Tendencia base (ligero crecimiento)
        trend = 1 + 0.002 * months

        # Datos por mes
        data = {
            "months": months,
            "clientes": [], "ingresos": [], "costes_mp": [],
            "costes_total": [], "beneficio": [], "ticket_medio": [],
            "ocupacion": [], "platos_vendidos": [],
        }

        base_clientes = max(self.datos.clientes_mes, 200)
        base_ingresos = max(self.datos.ingresos_totales, 5000)
        base_costes_mp = max(self.datos.coste_mp_total, 2000)
        base_costes_total = max(self.datos.costes_totales, 8000)

        for m in range(n_months):
            noise = np.random.normal(1.0, 0.08)
            factor = seasonal_full[m] * trend[m] * noise

            clientes = int(base_clientes * factor)
            ingresos = base_ingresos * factor * np.random.normal(1.0, 0.05)
            costes_mp = base_costes_mp * factor * np.random.normal(1.0, 0.06)
            costes_total = base_costes_total * (0.6 + 0.4 * factor) * np.random.normal(1.0, 0.04)

            data["clientes"].append(clientes)
            data["ingresos"].append(max(0, ingresos))
            data["costes_mp"].append(max(0, costes_mp))
            data["costes_total"].append(max(0, costes_total))
            data["beneficio"].append(ingresos - costes_total)
            data["ticket_medio"].append(ingresos / clientes if clientes > 0 else 0)
            data["ocupacion"].append(min(100, clientes / max(1, self.datos.asientos * self.datos.dias_apertura_mes * self.datos.servicios_por_dia) * 100))
            data["platos_vendidos"].append(int(sum(p.unidades_vendidas for p in self.datos.platos) * factor))

        # Convertir a numpy
        for key in data:
            data[key] = np.array(data[key])

        return data

    # =========================================================
    # 2. ENTRENAMIENTO DE MODELOS
    # =========================================================

    def train_models(self) -> Dict:
        """Entrena todos los modelos de ML con los datos del restaurante."""
        if not HAS_SKLEARN:
            return {"success": False, "error": "scikit-learn no está instalado"}

        data = self._generate_synthetic_data(36)  # 3 años sintéticos
        if not data:
            return {"success": False, "error": "No hay datos suficientes"}

        results = {}

        # --- Modelo 1: Predicción de ingresos ---
        try:
            X_rev = np.column_stack([
                data["months"],
                data["clientes"],
                data["ocupacion"],
                np.sin(2 * np.pi * data["months"] / 12),  # Estacionalidad
                np.cos(2 * np.pi * data["months"] / 12),
            ])
            y_rev = data["ingresos"]

            self.scaler_rev = StandardScaler()
            X_rev_scaled = self.scaler_rev.fit_transform(X_rev)

            self.models["revenue"] = GradientBoostingRegressor(
                n_estimators=100, max_depth=4, learning_rate=0.1, random_state=42
            )
            self.models["revenue"].fit(X_rev_scaled, y_rev)

            y_pred = self.models["revenue"].predict(X_rev_scaled)
            results["revenue_r2"] = r2_score(y_rev, y_pred)
            results["revenue_mae"] = mean_absolute_error(y_rev, y_pred)
        except Exception as e:
            results["revenue_error"] = str(e)

        # --- Modelo 2: Predicción de demanda (clientes) ---
        try:
            X_dem = np.column_stack([
                data["months"],
                np.sin(2 * np.pi * data["months"] / 12),
                np.cos(2 * np.pi * data["months"] / 12),
                data["months"] ** 2 / 1000,
            ])
            y_dem = data["clientes"]

            self.scaler_dem = StandardScaler()
            X_dem_scaled = self.scaler_dem.fit_transform(X_dem)

            self.models["demand"] = RandomForestRegressor(
                n_estimators=80, max_depth=5, random_state=42
            )
            self.models["demand"].fit(X_dem_scaled, y_dem)

            y_pred = self.models["demand"].predict(X_dem_scaled)
            results["demand_r2"] = r2_score(y_dem, y_pred)
        except Exception as e:
            results["demand_error"] = str(e)

        # --- Modelo 3: Clustering de platos ---
        try:
            if len(self.datos.platos) >= 4:
                X_platos = np.array([
                    [p.precio_venta, p.coste_mp, p.margen_bruto,
                     p.unidades_vendidas, p.pct_coste_mp]
                    for p in self.datos.platos
                ])
                scaler_p = StandardScaler()
                X_platos_scaled = scaler_p.fit_transform(X_platos)

                n_clusters = min(4, len(self.datos.platos))
                self.models["clustering"] = KMeans(n_clusters=n_clusters, random_state=42, n_init=10)
                self.models["clustering"].fit(X_platos_scaled)
                self.scaler_platos = scaler_p
                results["clustering_ok"] = True
        except Exception as e:
            results["clustering_error"] = str(e)

        # --- Modelo 4: Optimización de precios ---
        try:
            prices = np.array([p.precio_venta for p in self.datos.platos])
            costs = np.array([p.coste_mp for p in self.datos.platos])
            sales = np.array([p.unidades_vendidas for p in self.datos.platos])

            # Elasticidad precio-demanda simulada
            price_ratios = prices / (costs + 0.01)
            X_price = np.column_stack([price_ratios, costs, prices])
            y_price = sales

            self.models["price_opt"] = Ridge(alpha=1.0)
            self.models["price_opt"].fit(X_price, y_price)
            results["price_opt_ok"] = True
        except Exception as e:
            results["price_error"] = str(e)

        self.is_trained = True
        results["success"] = True
        results["models_trained"] = list(self.models.keys())
        return results

    # =========================================================
    # 3. PREDICCIONES
    # =========================================================

    def predict_revenue(self, months_ahead: int = 6) -> List[Dict]:
        """Predice ingresos para los próximos N meses."""
        if "revenue" not in self.models:
            return []

        predictions = []
        base_month = 36  # Después de los datos de entrenamiento
        base_clientes = max(self.datos.clientes_mes, 200)
        base_ocup = self.datos.ocupacion_media if hasattr(self.datos, 'ocupacion_media') else 60

        for i in range(months_ahead):
            m = base_month + i
            month_of_year = m % 12
            seasonal_factors = [0.85, 0.80, 0.90, 0.95, 1.00, 1.10, 1.15, 1.20, 1.05, 0.95, 0.90, 1.10]
            est_clientes = base_clientes * seasonal_factors[month_of_year] * (1 + 0.002 * m)

            X = np.array([[
                m, est_clientes, base_ocup,
                np.sin(2 * np.pi * m / 12),
                np.cos(2 * np.pi * m / 12),
            ]])
            X_scaled = self.scaler_rev.transform(X)
            pred = self.models["revenue"].predict(X_scaled)[0]

            now = datetime.now()
            pred_month = (now.month + i - 1) % 12 + 1
            pred_year = now.year + (now.month + i - 1) // 12
            month_names = ["Ene", "Feb", "Mar", "Abr", "May", "Jun",
                           "Jul", "Ago", "Sep", "Oct", "Nov", "Dic"]

            predictions.append({
                "mes": f"{month_names[pred_month - 1]} {pred_year}",
                "ingresos_predichos": max(0, pred),
                "clientes_estimados": int(est_clientes),
                "confianza": min(95, max(60, 90 - i * 3)),
            })

        return predictions

    def predict_demand(self, months_ahead: int = 6) -> List[Dict]:
        """Predice demanda (clientes) para los próximos N meses."""
        if "demand" not in self.models:
            return []

        predictions = []
        base_month = 36
        month_names = ["Ene", "Feb", "Mar", "Abr", "May", "Jun",
                       "Jul", "Ago", "Sep", "Oct", "Nov", "Dic"]

        for i in range(months_ahead):
            m = base_month + i
            X = np.array([[m, np.sin(2 * np.pi * m / 12),
                           np.cos(2 * np.pi * m / 12), m ** 2 / 1000]])
            X_scaled = self.scaler_dem.transform(X)
            pred = self.models["demand"].predict(X_scaled)[0]

            now = datetime.now()
            pred_month = (now.month + i - 1) % 12 + 1
            pred_year = now.year + (now.month + i - 1) // 12

            predictions.append({
                "mes": f"{month_names[pred_month - 1]} {pred_year}",
                "clientes_predichos": int(max(0, pred)),
                "confianza": min(95, max(60, 90 - i * 3)),
            })

        return predictions

    def optimize_prices(self) -> List[Dict]:
        """Sugiere precios óptimos para cada plato usando el modelo de elasticidad."""
        if "price_opt" not in self.models or not self.datos.platos:
            return []

        suggestions = []
        for plato in self.datos.platos:
            best_price = plato.precio_venta
            best_profit = plato.margen_bruto * plato.unidades_vendidas

            # Probar diferentes precios
            for delta_pct in np.arange(-0.20, 0.30, 0.02):
                test_price = plato.precio_venta * (1 + delta_pct)
                if test_price <= plato.coste_mp * 1.1:
                    continue

                price_ratio = test_price / (plato.coste_mp + 0.01)
                X_test = np.array([[price_ratio, plato.coste_mp, test_price]])
                pred_sales = self.models["price_opt"].predict(X_test)[0]
                pred_sales = max(0, pred_sales)

                profit = (test_price - plato.coste_mp) * pred_sales
                if profit > best_profit:
                    best_profit = profit
                    best_price = test_price

            cambio_pct = ((best_price - plato.precio_venta) / plato.precio_venta) * 100
            suggestions.append({
                "plato": plato.nombre,
                "precio_actual": plato.precio_venta,
                "precio_sugerido": round(best_price, 2),
                "cambio_pct": round(cambio_pct, 1),
                "beneficio_actual": plato.margen_bruto * plato.unidades_vendidas,
                "beneficio_estimado": round(best_profit, 2),
                "mejora_pct": round(((best_profit - plato.margen_bruto * plato.unidades_vendidas) /
                                     max(1, plato.margen_bruto * plato.unidades_vendidas)) * 100, 1),
            })

        return sorted(suggestions, key=lambda x: x["mejora_pct"], reverse=True)

    def cluster_analysis(self) -> List[Dict]:
        """Análisis de clustering de platos (agrupación inteligente)."""
        if "clustering" not in self.models or not self.datos.platos:
            return []

        X_platos = np.array([
            [p.precio_venta, p.coste_mp, p.margen_bruto,
             p.unidades_vendidas, p.pct_coste_mp]
            for p in self.datos.platos
        ])
        X_scaled = self.scaler_platos.transform(X_platos)
        labels = self.models["clustering"].predict(X_scaled)

        cluster_names = ["Grupo Premium", "Grupo Eficiente", "Grupo Popular", "Grupo a Revisar"]
        results = []
        for i, plato in enumerate(self.datos.platos):
            results.append({
                "plato": plato.nombre,
                "cluster": int(labels[i]),
                "cluster_nombre": cluster_names[labels[i] % len(cluster_names)],
                "precio": plato.precio_venta,
                "margen": plato.margen_bruto,
                "ventas": plato.unidades_vendidas,
            })
        return results

    def health_score(self) -> Dict:
        """Calcula un score de salud financiera 0-100 con IA."""
        if not self.datos:
            return {"score": 0, "details": []}

        scores = []
        details = []

        # Factor 1: Margen neto (0-25 puntos)
        ingresos = self.datos.ingresos_totales
        costes = self.datos.costes_totales
        margen = ((ingresos - costes) / ingresos * 100) if ingresos > 0 else 0
        s1 = min(25, max(0, margen * 1.5))
        scores.append(s1)
        details.append({"factor": "Margen neto", "score": round(s1, 1), "max": 25,
                         "valor": f"{margen:.1f}%", "ideal": ">12%"})

        # Factor 2: % Materia prima (0-20 puntos)
        pct_mp = (self.datos.coste_mp_total / ingresos * 100) if ingresos > 0 else 100
        s2 = max(0, 20 - abs(pct_mp - 35) * 0.8)
        scores.append(s2)
        details.append({"factor": "% Materia prima", "score": round(s2, 1), "max": 20,
                         "valor": f"{pct_mp:.1f}%", "ideal": "30-38%"})

        # Factor 3: Diversificación carta (0-15 puntos)
        if self.datos.platos:
            n_platos = len(self.datos.platos)
            cats = set(p.categoria for p in self.datos.platos)
            s3 = min(15, n_platos * 0.8 + len(cats) * 2)
        else:
            s3 = 0
        scores.append(s3)
        details.append({"factor": "Diversificación carta", "score": round(s3, 1), "max": 15,
                         "valor": f"{len(self.datos.platos)} platos", "ideal": ">15 platos"})

        # Factor 4: Ocupación (0-20 puntos)
        ocup = self.datos.ocupacion_media if hasattr(self.datos, 'ocupacion_media') else 0
        s4 = min(20, ocup * 0.25)
        scores.append(s4)
        details.append({"factor": "Ocupación media", "score": round(s4, 1), "max": 20,
                         "valor": f"{ocup:.0f}%", "ideal": ">70%"})

        # Factor 5: Cash flow (0-20 puntos)
        cf = (ingresos - costes) + self.datos.amortizaciones
        cf_ratio = cf / ingresos * 100 if ingresos > 0 else 0
        s5 = min(20, max(0, cf_ratio * 1.2))
        scores.append(s5)
        details.append({"factor": "Cash flow", "score": round(s5, 1), "max": 20,
                         "valor": f"{cf:,.0f}€", "ideal": ">10% ingresos"})

        total = sum(scores)
        if total >= 80:
            nivel = "Excelente"
        elif total >= 60:
            nivel = "Bueno"
        elif total >= 40:
            nivel = "Regular"
        elif total >= 20:
            nivel = "Débil"
        else:
            nivel = "Crítico"

        return {
            "score": round(total, 1),
            "nivel": nivel,
            "details": details,
            "recomendaciones": self._ai_recommendations(details, total),
        }

    def _ai_recommendations(self, details: List[Dict], score: float) -> List[str]:
        """Genera recomendaciones basadas en el scoring IA."""
        recs = []
        for d in details:
            ratio = d["score"] / d["max"]
            if ratio < 0.4:
                recs.append(f"🔴 {d['factor']}: {d['valor']} (ideal: {d['ideal']}). Prioridad ALTA.")
            elif ratio < 0.7:
                recs.append(f"🟡 {d['factor']}: {d['valor']} (ideal: {d['ideal']}). Mejorable.")

        if score < 40:
            recs.append("⚠️ Puntuación global baja. Se recomienda revisión urgente de costes y carta.")
        return recs

    def whatif_simulation(self, changes: Dict) -> Dict:
        """
        Simulación what-if: ¿qué pasa si cambio X?
        changes puede incluir: precio_cambio_pct, costes_mp_cambio_pct,
        clientes_cambio_pct, personal_cambio_pct
        """
        ingresos_base = self.datos.ingresos_totales
        costes_base = self.datos.costes_totales
        cf_base = self.datos.costes_fijos_total
        cv_base = self.datos.costes_variables_total

        factor_precio = 1 + changes.get("precio_cambio_pct", 0) / 100
        factor_mp = 1 + changes.get("costes_mp_cambio_pct", 0) / 100
        factor_clientes = 1 + changes.get("clientes_cambio_pct", 0) / 100
        factor_personal = 1 + changes.get("personal_cambio_pct", 0) / 100

        # Elasticidad: si subes precios, pierdes algo de clientes
        elasticidad = -0.3  # Típica hostelería
        clientes_ajustados = factor_clientes * (1 + elasticidad * (factor_precio - 1))

        nuevo_ingresos = ingresos_base * factor_precio * clientes_ajustados
        nuevo_mp = self.datos.coste_mp_total * factor_mp * clientes_ajustados
        nuevo_personal = (self.datos.salarios_fijos + self.datos.seguridad_social) * factor_personal + self.datos.salarios_variables * clientes_ajustados
        nuevo_cf = self.datos.alquiler + self.datos.seguros + self.datos.suministros_fijos + self.datos.amortizaciones + (self.datos.salarios_fijos + self.datos.seguridad_social) * factor_personal
        nuevo_cv = nuevo_mp + self.datos.salarios_variables * clientes_ajustados + self.datos.suministros_variables * clientes_ajustados
        nuevo_costes = nuevo_cf + nuevo_cv
        nuevo_beneficio = nuevo_ingresos - nuevo_costes

        return {
            "ingresos_antes": ingresos_base,
            "ingresos_despues": nuevo_ingresos,
            "costes_antes": costes_base,
            "costes_despues": nuevo_costes,
            "beneficio_antes": ingresos_base - costes_base,
            "beneficio_despues": nuevo_beneficio,
            "margen_antes": ((ingresos_base - costes_base) / ingresos_base * 100) if ingresos_base > 0 else 0,
            "margen_despues": (nuevo_beneficio / nuevo_ingresos * 100) if nuevo_ingresos > 0 else 0,
            "impacto": nuevo_beneficio - (ingresos_base - costes_base),
        }
