"""
Registro Diario — Convastro Pro
Registra clientes, platos, costes y servicios día a día.
La IA aprende del histórico para dar recomendaciones personalizadas.
"""

import json
import os
from datetime import datetime, timedelta
from typing import Dict, List, Optional

try:
    import numpy as np
    from sklearn.linear_model import LinearRegression
    HAS_ML = True
except ImportError:
    HAS_ML = False


class DailyRegister:
    """Gestiona el registro diario y genera insights con IA."""

    def __init__(self, data_dir: str = None):
        if data_dir:
            self.data_dir = data_dir
        else:
            self.data_dir = os.path.join(os.path.expanduser("~"), ".convastro", "daily")
        os.makedirs(self.data_dir, exist_ok=True)
        self.data_file = os.path.join(self.data_dir, "registro.json")
        self.registros: List[Dict] = self._load()

    def _load(self) -> List[Dict]:
        if os.path.exists(self.data_file):
            try:
                with open(self.data_file, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except Exception:
                return []
        return []

    def _save(self):
        with open(self.data_file, 'w', encoding='utf-8') as f:
            json.dump(self.registros, f, indent=2, ensure_ascii=False)

    def add_entry(self, fecha: str, datos: Dict) -> bool:
        """
        Añade registro de un día.
        datos = {
            "clientes_comida": 45,
            "clientes_cena": 38,
            "ingresos": 2340.50,
            "costes_mp": 680.00,
            "costes_personal": 520.00,
            "platos_vendidos": [
                {"nombre": "Solomillo", "cantidad": 12, "precio": 18.0, "coste": 6.5},
                {"nombre": "Ensalada", "cantidad": 22, "precio": 10.0, "coste": 2.5},
            ],
            "dia_semana": "lunes",
            "clima": "soleado",
            "evento_especial": "",
            "notas": "Día tranquilo",
        }
        """
        entry = {
            "fecha": fecha,
            "ts": datetime.now().isoformat(),
            **datos,
            "clientes_total": datos.get("clientes_comida", 0) + datos.get("clientes_cena", 0),
            "beneficio": datos.get("ingresos", 0) - datos.get("costes_mp", 0) - datos.get("costes_personal", 0),
        }

        # Si ya existe entrada para esa fecha, actualizar
        for i, r in enumerate(self.registros):
            if r["fecha"] == fecha:
                self.registros[i] = entry
                self._save()
                return True

        self.registros.append(entry)
        self.registros.sort(key=lambda x: x["fecha"])
        self._save()
        return True

    def get_entries(self, days: int = 30) -> List[Dict]:
        """Últimos N días de registros."""
        if days <= 0:
            return self.registros
        cutoff = (datetime.now() - timedelta(days=days)).strftime("%Y-%m-%d")
        return [r for r in self.registros if r["fecha"] >= cutoff]

    def get_entry(self, fecha: str) -> Optional[Dict]:
        for r in self.registros:
            if r["fecha"] == fecha:
                return r
        return None

    def get_summary(self, days: int = 30) -> Dict:
        """Resumen del período."""
        entries = self.get_entries(days)
        if not entries:
            return {"error": "No hay datos registrados"}

        n = len(entries)
        total_clientes = sum(e.get("clientes_total", 0) for e in entries)
        total_ingresos = sum(e.get("ingresos", 0) for e in entries)
        total_costes_mp = sum(e.get("costes_mp", 0) for e in entries)
        total_beneficio = sum(e.get("beneficio", 0) for e in entries)

        # Mejor y peor día
        by_ingresos = sorted(entries, key=lambda x: x.get("ingresos", 0))
        best = by_ingresos[-1] if by_ingresos else None
        worst = by_ingresos[0] if by_ingresos else None

        # Por día de la semana
        dias_semana = {}
        for e in entries:
            d = e.get("dia_semana", "desconocido")
            if d not in dias_semana:
                dias_semana[d] = {"clientes": 0, "ingresos": 0, "count": 0}
            dias_semana[d]["clientes"] += e.get("clientes_total", 0)
            dias_semana[d]["ingresos"] += e.get("ingresos", 0)
            dias_semana[d]["count"] += 1

        for d in dias_semana:
            c = dias_semana[d]["count"]
            dias_semana[d]["media_clientes"] = dias_semana[d]["clientes"] / c if c > 0 else 0
            dias_semana[d]["media_ingresos"] = dias_semana[d]["ingresos"] / c if c > 0 else 0

        # Plato más vendido del período
        platos_total = {}
        for e in entries:
            for p in e.get("platos_vendidos", []):
                nombre = p.get("nombre", "")
                if nombre not in platos_total:
                    platos_total[nombre] = {"cantidad": 0, "ingresos": 0, "coste": 0}
                platos_total[nombre]["cantidad"] += p.get("cantidad", 0)
                platos_total[nombre]["ingresos"] += p.get("cantidad", 0) * p.get("precio", 0)
                platos_total[nombre]["coste"] += p.get("cantidad", 0) * p.get("coste", 0)

        top_platos = sorted(platos_total.items(), key=lambda x: x[1]["cantidad"], reverse=True)

        return {
            "periodo_dias": days,
            "dias_registrados": n,
            "total_clientes": total_clientes,
            "media_clientes_dia": total_clientes / n if n > 0 else 0,
            "total_ingresos": total_ingresos,
            "media_ingresos_dia": total_ingresos / n if n > 0 else 0,
            "total_costes_mp": total_costes_mp,
            "pct_mp": (total_costes_mp / total_ingresos * 100) if total_ingresos > 0 else 0,
            "total_beneficio": total_beneficio,
            "media_beneficio_dia": total_beneficio / n if n > 0 else 0,
            "ticket_medio": total_ingresos / total_clientes if total_clientes > 0 else 0,
            "mejor_dia": {"fecha": best["fecha"], "ingresos": best["ingresos"]} if best else None,
            "peor_dia": {"fecha": worst["fecha"], "ingresos": worst["ingresos"]} if worst else None,
            "por_dia_semana": dias_semana,
            "top_platos": top_platos[:10],
        }

    # ============================================
    # IA: ANÁLISIS Y PREDICCIONES BASADAS EN HISTÓRICO
    # ============================================

    def ai_trends(self, days: int = 60) -> Dict:
        """Analiza tendencias con regresión lineal."""
        if not HAS_ML:
            return {"error": "scikit-learn no disponible"}

        entries = self.get_entries(days)
        if len(entries) < 7:
            return {"error": "Se necesitan al menos 7 días de datos"}

        X = np.arange(len(entries)).reshape(-1, 1)
        y_ingresos = np.array([e.get("ingresos", 0) for e in entries])
        y_clientes = np.array([e.get("clientes_total", 0) for e in entries])

        # Tendencia ingresos
        model_ing = LinearRegression().fit(X, y_ingresos)
        slope_ing = model_ing.coef_[0]

        # Tendencia clientes
        model_cli = LinearRegression().fit(X, y_clientes)
        slope_cli = model_cli.coef_[0]

        # Predicción próximos 7 días
        X_pred = np.arange(len(entries), len(entries) + 7).reshape(-1, 1)
        pred_ingresos = model_ing.predict(X_pred)
        pred_clientes = model_cli.predict(X_pred)

        # Día más rentable
        best_weekday = None
        if entries:
            day_profits = {}
            for e in entries:
                d = e.get("dia_semana", "")
                if d:
                    day_profits.setdefault(d, []).append(e.get("beneficio", 0))
            best_weekday = max(day_profits.items(),
                               key=lambda x: sum(x[1]) / len(x[1]) if x[1] else 0,
                               default=("", []))[0]

        return {
            "tendencia_ingresos": "↑ Subiendo" if slope_ing > 5 else "↓ Bajando" if slope_ing < -5 else "→ Estable",
            "cambio_diario_ingresos": round(slope_ing, 2),
            "tendencia_clientes": "↑ Subiendo" if slope_cli > 0.5 else "↓ Bajando" if slope_cli < -0.5 else "→ Estable",
            "cambio_diario_clientes": round(slope_cli, 2),
            "prediccion_7_dias": {
                "ingresos_total": round(float(pred_ingresos.sum()), 2),
                "ingresos_medio": round(float(pred_ingresos.mean()), 2),
                "clientes_total": int(pred_clientes.sum()),
            },
            "dia_mas_rentable": best_weekday,
            "datos_analizados": len(entries),
        }

    def ai_recommendations(self) -> List[str]:
        """Genera recomendaciones basadas en el histórico."""
        summary = self.get_summary(30)
        if "error" in summary:
            return ["Registra datos diarios para recibir recomendaciones personalizadas."]

        recs = []

        # Tendencia
        trends = self.ai_trends(30)
        if "error" not in trends:
            if "Bajando" in trends.get("tendencia_ingresos", ""):
                recs.append(f"📉 Tus ingresos bajan ~{abs(trends['cambio_diario_ingresos']):.0f}€/día. "
                            f"Revisa carta y marketing.")
            if "Subiendo" in trends.get("tendencia_clientes", ""):
                recs.append(f"📈 Más clientes cada día (+{trends['cambio_diario_clientes']:.1f}/día). ¡Sigue así!")

        # % MP
        if summary.get("pct_mp", 0) > 40:
            recs.append(f"⚠️ Tu % de materia prima es {summary['pct_mp']:.1f}%. Objetivo: <38%.")
        elif summary.get("pct_mp", 0) < 30:
            recs.append(f"✅ Excelente control de MP ({summary['pct_mp']:.1f}%).")

        # Mejor día
        ds = summary.get("por_dia_semana", {})
        if ds:
            mejor = max(ds.items(), key=lambda x: x[1]["media_ingresos"], default=None)
            peor = min(ds.items(), key=lambda x: x[1]["media_ingresos"], default=None)
            if mejor and peor:
                recs.append(f"📊 Mejor día: {mejor[0]} ({mejor[1]['media_ingresos']:,.0f}€). "
                            f"Peor: {peor[0]} ({peor[1]['media_ingresos']:,.0f}€). "
                            f"Lanza ofertas el {peor[0]}.")

        # Platos
        top = summary.get("top_platos", [])
        if top:
            best_plato = top[0]
            recs.append(f"⭐ Plato estrella: {best_plato[0]} ({best_plato[1]['cantidad']} uds). Mantén stock.")
            if len(top) > 5:
                worst = top[-1]
                recs.append(f"🐕 Menos vendido: {worst[0]} ({worst[1]['cantidad']} uds). ¿Eliminar o reformular?")

        if not recs:
            recs.append("✅ Sigue registrando datos diarios. Más datos = mejores recomendaciones.")

        return recs
