"""
Módulo de exportación de informes — RestaurantAI Pro
Genera informes en PDF y Excel profesionales.
"""

import os
from datetime import datetime
from typing import Dict, Optional

try:
    from reportlab.lib import colors as rl_colors
    from reportlab.lib.pagesizes import A4
    from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
    from reportlab.lib.units import mm, cm
    from reportlab.platypus import (
        SimpleDocTemplate, Paragraph, Spacer, Table, TableStyle,
        PageBreak, Image as RLImage
    )
    from reportlab.lib.enums import TA_CENTER, TA_LEFT, TA_RIGHT
    HAS_REPORTLAB = True
except ImportError:
    HAS_REPORTLAB = False

try:
    from openpyxl import Workbook
    from openpyxl.styles import Font, Alignment, PatternFill, Border, Side
    from openpyxl.chart import PieChart, BarChart, Reference
    HAS_OPENPYXL = True
except ImportError:
    HAS_OPENPYXL = False


class ReportExporter:
    """Exporta informes financieros a PDF y Excel."""

    def __init__(self, datos_restaurante, analisis):
        self.datos = datos_restaurante
        self.analisis = analisis

    # =========================================
    # EXPORTACIÓN PDF
    # =========================================

    def export_pdf(self, filepath: str, logo_path: Optional[str] = None) -> bool:
        """Genera informe completo en PDF."""
        if not HAS_REPORTLAB:
            return False

        try:
            doc = SimpleDocTemplate(
                filepath, pagesize=A4,
                rightMargin=2 * cm, leftMargin=2 * cm,
                topMargin=2 * cm, bottomMargin=2 * cm
            )

            styles = getSampleStyleSheet()
            # Estilos personalizados
            styles.add(ParagraphStyle(
                name='TitleCustom', fontSize=24, spaceAfter=20,
                textColor=rl_colors.HexColor('#1a1a2e'),
                fontName='Helvetica-Bold', alignment=TA_CENTER
            ))
            styles.add(ParagraphStyle(
                name='SubtitleCustom', fontSize=14, spaceAfter=12,
                textColor=rl_colors.HexColor('#6b7b8d'),
                fontName='Helvetica', alignment=TA_CENTER
            ))
            styles.add(ParagraphStyle(
                name='SectionTitle', fontSize=16, spaceBefore=20, spaceAfter=10,
                textColor=rl_colors.HexColor('#e94560'),
                fontName='Helvetica-Bold'
            ))
            styles.add(ParagraphStyle(
                name='BodyCustom', fontSize=11, spaceAfter=8,
                textColor=rl_colors.HexColor('#333333'),
                fontName='Helvetica', leading=14
            ))

            elements = []

            # PORTADA
            elements.append(Spacer(1, 3 * cm))

            if logo_path and os.path.exists(logo_path):
                try:
                    img = RLImage(logo_path, width=6 * cm, height=2 * cm)
                    elements.append(img)
                    elements.append(Spacer(1, 1 * cm))
                except Exception:
                    pass

            elements.append(Paragraph("RestaurantAI Pro", styles['TitleCustom']))
            elements.append(Paragraph("Informe de Análisis Financiero", styles['SubtitleCustom']))
            elements.append(Spacer(1, 1 * cm))
            elements.append(Paragraph(
                f"<b>{self.datos.nombre_restaurante}</b>", styles['SubtitleCustom']
            ))
            elements.append(Paragraph(
                f"Fecha: {datetime.now().strftime('%d/%m/%Y')}", styles['SubtitleCustom']
            ))
            elements.append(Spacer(1, 2 * cm))
            elements.append(Paragraph(
                "© 2026 Roberto Martín Gutiérrez &amp; Inés Fraile Verdugo",
                styles['SubtitleCustom']
            ))
            elements.append(PageBreak())

            # ÍNDICE
            elements.append(Paragraph("Índice", styles['SectionTitle']))
            toc_items = [
                "1. Resumen Ejecutivo",
                "2. Estructura de Costes",
                "3. Punto Muerto / Umbral de Rentabilidad",
                "4. Menu Engineering",
                "5. Análisis a Corto Plazo",
                "6. Análisis a Medio Plazo",
                "7. Análisis a Largo Plazo",
                "8. Plan de Acción",
            ]
            for item in toc_items:
                elements.append(Paragraph(item, styles['BodyCustom']))
            elements.append(PageBreak())

            # 1. RESUMEN EJECUTIVO
            costes = self.analisis.analisis_costes()
            cp = self.analisis.analisis_corto_plazo()

            elements.append(Paragraph("1. Resumen Ejecutivo", styles['SectionTitle']))
            elements.append(Paragraph(
                f"El restaurante <b>{self.datos.nombre_restaurante}</b> "
                f"({self.datos.tipo_restaurante}) presenta los siguientes indicadores "
                f"financieros principales:", styles['BodyCustom']
            ))

            resumen_data = [
                ["Indicador", "Valor", "Estado"],
                ["Ingresos mensuales", f"{costes['ingresos']:,.2f}€", "—"],
                ["Costes totales", f"{costes['costes_totales']:,.2f}€", "—"],
                ["Beneficio neto", f"{costes['beneficio_neto']:,.2f}€",
                 "✓" if costes['beneficio_neto'] > 0 else "✗"],
                ["Margen neto", f"{costes['margen_neto_pct']:.1f}%",
                 "✓" if costes['margen_neto_pct'] > 10 else "!"],
                ["Ticket medio", f"{cp['ticket_medio']:.2f}€", "—"],
                ["Cash flow mensual", f"{cp['cash_flow_mensual']:,.2f}€",
                 "✓" if cp['cash_flow_mensual'] > 0 else "✗"],
                ["% Materia prima", f"{costes['pct_materia_prima']:.1f}%",
                 "✓" if costes['ref_mp_ok'] else "!"],
                ["% Personal", f"{costes['pct_personal']:.1f}%",
                 "✓" if costes['ref_personal_ok'] else "!"],
            ]

            t = Table(resumen_data, colWidths=[7 * cm, 5 * cm, 3 * cm])
            t.setStyle(TableStyle([
                ('BACKGROUND', (0, 0), (-1, 0), rl_colors.HexColor('#1a1a2e')),
                ('TEXTCOLOR', (0, 0), (-1, 0), rl_colors.white),
                ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                ('FONTSIZE', (0, 0), (-1, -1), 10),
                ('ALIGN', (1, 0), (-1, -1), 'CENTER'),
                ('GRID', (0, 0), (-1, -1), 0.5, rl_colors.HexColor('#cccccc')),
                ('ROWBACKGROUNDS', (0, 1), (-1, -1),
                 [rl_colors.white, rl_colors.HexColor('#f5f5f5')]),
                ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
                ('TOPPADDING', (0, 0), (-1, -1), 6),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 6),
            ]))
            elements.append(t)
            elements.append(Spacer(1, 0.5 * cm))

            # Alertas
            if costes.get("alertas"):
                elements.append(Paragraph("<b>Alertas:</b>", styles['BodyCustom']))
                for alerta in costes["alertas"]:
                    elements.append(Paragraph(f"• {alerta}", styles['BodyCustom']))

            elements.append(PageBreak())

            # 2. ESTRUCTURA DE COSTES
            elements.append(Paragraph("2. Estructura de Costes", styles['SectionTitle']))

            costes_data = [
                ["Concepto", "Importe (€)", "% sobre ingresos"],
                ["COSTES FIJOS", "", ""],
                ["Alquiler", f"{self.datos.alquiler:,.2f}", ""],
                ["Salarios fijos", f"{self.datos.salarios_fijos:,.2f}", ""],
                ["Seguridad Social", f"{self.datos.seguridad_social:,.2f}", ""],
                ["Seguros", f"{self.datos.seguros:,.2f}", ""],
                ["Suministros fijos", f"{self.datos.suministros_fijos:,.2f}", ""],
                ["Amortizaciones", f"{self.datos.amortizaciones:,.2f}", ""],
                ["Total fijos", f"{self.datos.costes_fijos_total:,.2f}",
                 f"{costes['pct_costes_fijos']:.1f}%"],
                ["COSTES VARIABLES", "", ""],
                ["Materia prima", f"{self.datos.coste_mp_total:,.2f}",
                 f"{costes['pct_materia_prima']:.1f}%"],
                ["Salarios variables", f"{self.datos.salarios_variables:,.2f}", ""],
                ["Suministros variables", f"{self.datos.suministros_variables:,.2f}", ""],
                ["Total variables", f"{self.datos.costes_variables_total:,.2f}",
                 f"{costes['pct_costes_variables']:.1f}%"],
                ["TOTAL COSTES", f"{self.datos.costes_totales:,.2f}",
                 f"{costes['pct_costes_totales']:.1f}%"],
            ]

            t2 = Table(costes_data, colWidths=[7 * cm, 4 * cm, 4 * cm])
            t2.setStyle(TableStyle([
                ('BACKGROUND', (0, 0), (-1, 0), rl_colors.HexColor('#1a1a2e')),
                ('TEXTCOLOR', (0, 0), (-1, 0), rl_colors.white),
                ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                ('BACKGROUND', (0, 1), (-1, 1), rl_colors.HexColor('#e8e8e8')),
                ('BACKGROUND', (0, 9), (-1, 9), rl_colors.HexColor('#e8e8e8')),
                ('BACKGROUND', (0, -1), (-1, -1), rl_colors.HexColor('#d4a574')),
                ('FONTNAME', (0, 1), (0, 1), 'Helvetica-Bold'),
                ('FONTNAME', (0, 8), (0, 8), 'Helvetica-Bold'),
                ('FONTNAME', (0, 9), (0, 9), 'Helvetica-Bold'),
                ('FONTNAME', (0, 13), (0, 13), 'Helvetica-Bold'),
                ('FONTNAME', (0, -1), (-1, -1), 'Helvetica-Bold'),
                ('FONTSIZE', (0, 0), (-1, -1), 10),
                ('ALIGN', (1, 0), (-1, -1), 'RIGHT'),
                ('GRID', (0, 0), (-1, -1), 0.5, rl_colors.HexColor('#cccccc')),
                ('VALIGN', (0, 0), (-1, -1), 'MIDDLE'),
                ('TOPPADDING', (0, 0), (-1, -1), 5),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 5),
            ]))
            elements.append(t2)
            elements.append(PageBreak())

            # 4. MENU ENGINEERING
            me = self.analisis.menu_engineering()
            elements.append(Paragraph("4. Menu Engineering", styles['SectionTitle']))

            if "clasificacion" in me:
                elements.append(Paragraph(
                    f"Análisis de {len(me['clasificacion'])} platos. "
                    f"Margen medio: {me['margen_medio']:.2f}€. "
                    f"Umbral de popularidad: {me['ip_umbral']:.1f}%.",
                    styles['BodyCustom']
                ))

                me_data = [["Plato", "Categoría", "PV(€)", "Margen(€)", "Uds", "IP(%)", "IM"]]
                for item in me["clasificacion"]:
                    me_data.append([
                        item["plato"][:25],
                        item["categoria_info"]["label"],
                        f"{item['precio']:.2f}",
                        f"{item['margen_bruto']:.2f}",
                        str(item["unidades"]),
                        f"{item['indice_popularidad']:.1f}",
                        f"{item['indice_margen']:.2f}",
                    ])

                t_me = Table(me_data, colWidths=[4.5 * cm, 3 * cm, 2 * cm, 2.5 * cm, 1.5 * cm, 1.5 * cm, 1.5 * cm])
                t_me.setStyle(TableStyle([
                    ('BACKGROUND', (0, 0), (-1, 0), rl_colors.HexColor('#1a1a2e')),
                    ('TEXTCOLOR', (0, 0), (-1, 0), rl_colors.white),
                    ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                    ('FONTSIZE', (0, 0), (-1, -1), 9),
                    ('ALIGN', (2, 0), (-1, -1), 'CENTER'),
                    ('GRID', (0, 0), (-1, -1), 0.5, rl_colors.HexColor('#cccccc')),
                    ('ROWBACKGROUNDS', (0, 1), (-1, -1),
                     [rl_colors.white, rl_colors.HexColor('#f5f5f5')]),
                    ('TOPPADDING', (0, 0), (-1, -1), 4),
                    ('BOTTOMPADDING', (0, 0), (-1, -1), 4),
                ]))
                elements.append(t_me)
                elements.append(Spacer(1, 0.5 * cm))

                # Recomendaciones
                if me.get("recomendaciones"):
                    elements.append(Paragraph("<b>Recomendaciones:</b>", styles['BodyCustom']))
                    for rec in me["recomendaciones"]:
                        clean_rec = rec.replace("⭐", "*").replace("🐄", "").replace("🧩", "").replace("🐕", "")
                        elements.append(Paragraph(f"• {clean_rec}", styles['BodyCustom']))

            elements.append(PageBreak())

            # 7. LARGO PLAZO
            lp = self.analisis.analisis_largo_plazo()
            elements.append(Paragraph("7. Análisis a Largo Plazo", styles['SectionTitle']))

            lp_data = [
                ["Indicador", "Valor"],
                ["Beneficio anual estimado", f"{lp['beneficio_anual']:,.2f}€"],
                ["Rentabilidad económica", f"{lp['rentabilidad_economica']:.1f}%"],
                ["ROI", f"{lp['roi']:.1f}%"],
                ["Cash flow anual", f"{lp['cash_flow_anual']:,.2f}€"],
                ["Payback estimado", f"{lp['payback_meses']:.0f} meses" if lp['payback_meses'] < 999 else "N/A"],
                ["Salud financiera", lp['salud_financiera']['estado'].upper()],
            ]

            t_lp = Table(lp_data, colWidths=[8 * cm, 6 * cm])
            t_lp.setStyle(TableStyle([
                ('BACKGROUND', (0, 0), (-1, 0), rl_colors.HexColor('#1a1a2e')),
                ('TEXTCOLOR', (0, 0), (-1, 0), rl_colors.white),
                ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                ('FONTSIZE', (0, 0), (-1, -1), 11),
                ('ALIGN', (1, 0), (-1, -1), 'RIGHT'),
                ('GRID', (0, 0), (-1, -1), 0.5, rl_colors.HexColor('#cccccc')),
                ('ROWBACKGROUNDS', (0, 1), (-1, -1),
                 [rl_colors.white, rl_colors.HexColor('#f5f5f5')]),
                ('TOPPADDING', (0, 0), (-1, -1), 6),
                ('BOTTOMPADDING', (0, 0), (-1, -1), 6),
            ]))
            elements.append(t_lp)
            elements.append(Spacer(1, 0.5 * cm))

            # Plan de acción
            elements.append(Paragraph("8. Plan de Acción", styles['SectionTitle']))
            for accion in lp.get("plan_accion", []):
                elements.append(Paragraph(accion, styles['BodyCustom']))

            # Footer
            elements.append(Spacer(1, 2 * cm))
            elements.append(Paragraph(
                "Informe generado por RestaurantAI Pro — © 2026 Roberto Martín Gutiérrez &amp; Inés Fraile Verdugo",
                ParagraphStyle('Footer', fontSize=8, textColor=rl_colors.HexColor('#999999'),
                               alignment=TA_CENTER)
            ))

            doc.build(elements)
            return True

        except Exception as e:
            print(f"Error exportando PDF: {e}")
            return False

    # =========================================
    # EXPORTACIÓN EXCEL
    # =========================================

    def export_excel(self, filepath: str) -> bool:
        """Genera informe en Excel con múltiples hojas."""
        if not HAS_OPENPYXL:
            return False

        try:
            wb = Workbook()

            # Estilos
            header_font = Font(name='Calibri', bold=True, size=12, color='FFFFFF')
            header_fill = PatternFill(start_color='1A1A2E', end_color='1A1A2E', fill_type='solid')
            gold_fill = PatternFill(start_color='D4A574', end_color='D4A574', fill_type='solid')
            border = Border(
                left=Side(style='thin', color='CCCCCC'),
                right=Side(style='thin', color='CCCCCC'),
                top=Side(style='thin', color='CCCCCC'),
                bottom=Side(style='thin', color='CCCCCC'),
            )

            def style_header(ws, row, max_col):
                for col in range(1, max_col + 1):
                    cell = ws.cell(row=row, column=col)
                    cell.font = header_font
                    cell.fill = header_fill
                    cell.alignment = Alignment(horizontal='center', vertical='center')
                    cell.border = border

            def style_cells(ws, start_row, end_row, max_col):
                for r in range(start_row, end_row + 1):
                    for c in range(1, max_col + 1):
                        cell = ws.cell(row=r, column=c)
                        cell.border = border
                        cell.alignment = Alignment(horizontal='center' if c > 1 else 'left')

            # HOJA 1: Resumen
            ws1 = wb.active
            ws1.title = "Resumen"
            costes = self.analisis.analisis_costes()
            cp = self.analisis.analisis_corto_plazo()

            ws1.append(["RestaurantAI Pro — Informe Financiero"])
            ws1.merge_cells('A1:D1')
            ws1.cell(1, 1).font = Font(name='Calibri', bold=True, size=16, color='1A1A2E')

            ws1.append([self.datos.nombre_restaurante])
            ws1.append([f"Fecha: {datetime.now().strftime('%d/%m/%Y')}"])
            ws1.append([])

            headers = ["Indicador", "Valor", "Referencia", "Estado"]
            ws1.append(headers)
            style_header(ws1, 5, 4)

            kpis = [
                ["Ingresos mensuales", costes['ingresos'], "—", "—"],
                ["Costes totales", costes['costes_totales'], "—", "—"],
                ["Beneficio neto", costes['beneficio_neto'], "—", "OK" if costes['beneficio_neto'] > 0 else "ALERTA"],
                ["Margen neto %", costes['margen_neto_pct'], ">10%", "OK" if costes['margen_neto_pct'] > 10 else "BAJO"],
                ["% Materia prima", costes['pct_materia_prima'], "38%", "OK" if costes['ref_mp_ok'] else "ALTO"],
                ["% Personal", costes['pct_personal'], "30%", "OK" if costes['ref_personal_ok'] else "ALTO"],
                ["Ticket medio", cp['ticket_medio'], "—", "—"],
                ["Cash flow mensual", cp['cash_flow_mensual'], ">0", "OK" if cp['cash_flow_mensual'] > 0 else "NEGATIVO"],
            ]

            for row_data in kpis:
                ws1.append(row_data)

            style_cells(ws1, 6, 5 + len(kpis), 4)

            for col in range(1, 5):
                ws1.column_dimensions[chr(64 + col)].width = 25

            # HOJA 2: Menu Engineering
            ws2 = wb.create_sheet("Menu Engineering")
            me = self.analisis.menu_engineering()

            ws2.append(["Menu Engineering — " + self.datos.nombre_restaurante])
            ws2.merge_cells('A1:G1')
            ws2.cell(1, 1).font = Font(name='Calibri', bold=True, size=14, color='1A1A2E')
            ws2.append([])

            me_headers = ["Plato", "Categoría", "PV (€)", "Coste MP (€)", "Margen (€)",
                          "Uds vendidas", "IP (%)", "IM", "Beneficio total"]
            ws2.append(me_headers)
            style_header(ws2, 3, 9)

            if "clasificacion" in me:
                for i, item in enumerate(me["clasificacion"]):
                    ws2.append([
                        item["plato"],
                        item["categoria_info"]["label"],
                        item["precio"],
                        item["coste_mp"],
                        item["margen_bruto"],
                        item["unidades"],
                        round(item["indice_popularidad"], 1),
                        round(item["indice_margen"], 2),
                        item["beneficio_total"],
                    ])

                style_cells(ws2, 4, 3 + len(me["clasificacion"]), 9)

            for col in range(1, 10):
                ws2.column_dimensions[chr(64 + col)].width = 16

            # HOJA 3: Costes detallados
            ws3 = wb.create_sheet("Costes Detallados")

            ws3.append(["Estructura de Costes — " + self.datos.nombre_restaurante])
            ws3.merge_cells('A1:C1')
            ws3.cell(1, 1).font = Font(name='Calibri', bold=True, size=14, color='1A1A2E')
            ws3.append([])

            ws3.append(["Concepto", "Importe (€)", "% sobre ingresos"])
            style_header(ws3, 3, 3)

            costes_rows = [
                ["COSTES FIJOS", "", ""],
                ["Alquiler", self.datos.alquiler, ""],
                ["Salarios fijos", self.datos.salarios_fijos, ""],
                ["Seguridad Social", self.datos.seguridad_social, ""],
                ["Seguros", self.datos.seguros, ""],
                ["Suministros fijos", self.datos.suministros_fijos, ""],
                ["Amortizaciones", self.datos.amortizaciones, ""],
                ["Total fijos", self.datos.costes_fijos_total, costes['pct_costes_fijos']],
                ["", "", ""],
                ["COSTES VARIABLES", "", ""],
                ["Materia prima", self.datos.coste_mp_total, costes['pct_materia_prima']],
                ["Salarios variables", self.datos.salarios_variables, ""],
                ["Suministros variables", self.datos.suministros_variables, ""],
                ["Total variables", self.datos.costes_variables_total, costes['pct_costes_variables']],
                ["", "", ""],
                ["TOTAL COSTES", self.datos.costes_totales, costes['pct_costes_totales']],
                ["INGRESOS", costes['ingresos'], 100],
                ["BENEFICIO NETO", costes['beneficio_neto'], costes['margen_neto_pct']],
            ]

            for row_data in costes_rows:
                ws3.append(row_data)

            style_cells(ws3, 4, 3 + len(costes_rows), 3)

            for col in range(1, 4):
                ws3.column_dimensions[chr(64 + col)].width = 25

            # HOJA 4: Proyección largo plazo
            ws4 = wb.create_sheet("Proyección")
            lp = self.analisis.analisis_largo_plazo()

            ws4.append(["Proyección a Largo Plazo — " + self.datos.nombre_restaurante])
            ws4.merge_cells('A1:D1')
            ws4.cell(1, 1).font = Font(name='Calibri', bold=True, size=14, color='1A1A2E')
            ws4.append([])

            ws4.append(["Año", "Ingresos (€)", "Costes (€)", "Beneficio (€)", "Margen (%)"])
            style_header(ws4, 3, 5)

            for p in lp.get("proyeccion_3_años", []):
                ws4.append([f"Año {p['año']}", p['ingresos'], p['costes'],
                            p['beneficio'], round(p['margen'], 1)])

            style_cells(ws4, 4, 3 + len(lp.get("proyeccion_3_años", [])), 5)

            for col in range(1, 6):
                ws4.column_dimensions[chr(64 + col)].width = 18

            wb.save(filepath)
            return True

        except Exception as e:
            print(f"Error exportando Excel: {e}")
            return False
