"""
Sistema de licencias para RestaurantAI Pro
Clave compartida fija para cifrado (funciona entre máquinas distintas).
Hardware ID solo se valida DENTRO del contenido de la licencia.
"""

import hashlib
import uuid
import json
import os
import base64
import hmac
from datetime import datetime, timedelta


class LicenseManager:
    """Gestiona licencias con clave simétrica fija."""

    LICENSE_FILE = "license.dat"
    _SECRET = "R3stAI_2026_RMG_IFV_pr0duct10n_k3y!"

    def __init__(self, app_dir=None):
        if app_dir:
            self.license_path = os.path.join(app_dir, self.LICENSE_FILE)
        else:
            self.license_path = os.path.join(
                os.path.expanduser("~"), ".restaurantai", self.LICENSE_FILE
            )
        os.makedirs(os.path.dirname(self.license_path), exist_ok=True)

    def get_hardware_id(self) -> str:
        """ID único legible: XXXX-XXXX-XXXX-XXXX"""
        try:
            mac = uuid.getnode()
            raw = f"{mac}-RestaurantAI-2026"
            h = hashlib.sha256(raw.encode()).hexdigest()[:16].upper()
            return f"{h[:4]}-{h[4:8]}-{h[8:12]}-{h[12:16]}"
        except Exception:
            return "0000-0000-0000-0000"

    def _sign(self, data: str) -> str:
        return hmac.new(self._SECRET.encode(), data.encode(), hashlib.sha256).hexdigest()[:32]

    def _encode(self, data: dict) -> str:
        json_str = json.dumps(data, separators=(',', ':'))
        b64 = base64.urlsafe_b64encode(json_str.encode()).decode().rstrip('=')
        sig = self._sign(json_str)
        return f"{b64}.{sig}"

    def _decode(self, key: str) -> dict:
        key = key.strip()
        parts = key.rsplit('.', 1)
        if len(parts) != 2:
            raise ValueError("Formato de clave inválido")
        b64_data, received_sig = parts
        padding = 4 - len(b64_data) % 4
        if padding != 4:
            b64_data += '=' * padding
        json_str = base64.urlsafe_b64decode(b64_data.encode()).decode()
        expected_sig = self._sign(json_str)
        if not hmac.compare_digest(received_sig, expected_sig):
            raise ValueError("Firma inválida")
        return json.loads(json_str)

    def generate_license_key(self, hardware_id: str, license_type: str = "pro",
                              duration_days: int = 365) -> str:
        """ADMIN: genera clave para un cliente."""
        data = {
            "hw": hardware_id,
            "t": license_type,
            "c": datetime.now().strftime("%Y%m%d"),
            "e": (datetime.now() + timedelta(days=duration_days)).strftime("%Y%m%d"),
        }
        return self._encode(data)

    def activate_license(self, license_key: str) -> dict:
        """Activa una licencia en este equipo."""
        try:
            data = self._decode(license_key)
        except ValueError as e:
            return {"success": False, "error": str(e)}
        except Exception:
            return {"success": False, "error": "Clave inválida o corrupta"}

        current_hw = self.get_hardware_id()
        license_hw = data.get("hw", "")
        if license_hw != current_hw:
            return {
                "success": False,
                "error": (f"Licencia para otro equipo.\n\n"
                          f"Licencia: {license_hw}\nEste equipo: {current_hw}\n\n"
                          f"Envía tu ID a convderover@gmail.com")
            }

        try:
            expires = datetime.strptime(data["e"], "%Y%m%d")
            if datetime.now() > expires:
                return {"success": False, "error": "Licencia expirada"}
        except Exception:
            return {"success": False, "error": "Fecha inválida en licencia"}

        with open(self.license_path, 'w') as f:
            json.dump({
                "key": license_key, "activated": datetime.now().isoformat(),
                "type": data.get("t", "pro"), "expires": data["e"], "hw": current_hw,
            }, f)

        return {
            "success": True, "type": data.get("t", "pro"),
            "expires": expires.strftime("%d/%m/%Y"),
            "days_left": (expires - datetime.now()).days,
        }

    def check_license(self) -> dict:
        if not os.path.exists(self.license_path):
            return {"valid": False, "type": "demo", "reason": "Sin licencia"}
        try:
            with open(self.license_path, 'r') as f:
                data = json.load(f)
            expires = datetime.strptime(data["expires"], "%Y%m%d")
            if datetime.now() > expires:
                return {"valid": False, "type": "demo", "reason": "Expirada"}
            current_hw = self.get_hardware_id()
            if data.get("hw") and data["hw"] != current_hw:
                return {"valid": False, "type": "demo", "reason": "Hardware distinto"}
            return {
                "valid": True, "type": data.get("type", "pro"),
                "expires": expires.strftime("%d/%m/%Y"),
                "days_left": (expires - datetime.now()).days,
            }
        except Exception:
            return {"valid": False, "type": "demo", "reason": "Corrupta"}

    def is_pro(self) -> bool:
        r = self.check_license()
        return r.get("valid", False) and r.get("type") == "pro"

    def deactivate(self):
        if os.path.exists(self.license_path):
            os.remove(self.license_path)
